(function ($) {
  if (typeof window.OTTO_DB === 'undefined') {
    return;
  }

  const settings = window.OTTO_DB;
  const groups = ['core', 'meta', 'acf', 'tax'];

  function escapeHtml(str) {
    return String(str)
      .replace(/&/g, '&amp;')
      .replace(/</g, '&lt;')
      .replace(/>/g, '&gt;')
      .replace(/"/g, '&quot;')
      .replace(/'/g, '&#039;');
  }

  function getCard($el) {
    return $el.closest('.otto-db-template-card');
  }

  function getIndex($card) {
    return $card.data('index');
  }

  function defaultStrings($card) {
    const $collapse = $card.find('.otto-db-collapse').first();
    return {
      title: $collapse.data('default-title') || settings.strings.untitled,
      sub: $collapse.data('default-sub') || settings.strings.unsaved,
    };
  }

  function currentPostType($card) {
    const $select = $card.find('.otto-db-pt-select');
    return {
      value: $.trim($select.val() || ''),
      label: $.trim($select.find('option:selected').text() || ''),
    };
  }

  function updateHeaderTexts($card) {
    const defaults = defaultStrings($card);
    const label = $.trim($card.find('.otto-db-label-input').val() || '');
    const slug = $.trim($card.find('.otto-db-slug-input').val() || '');
    const pt = currentPostType($card);
    const $title = $card.find('[data-role="template-title"]');
    const $sub = $card.find('[data-role="template-sub"]');

    $title.text(label || slug || defaults.title);
    $sub.text(slug ? `${slug} · ${pt.label}` : defaults.sub);
  }

  function buildDownloadBase(slug) {
    return settings.fileBase + encodeURIComponent(slug) + '.json';
  }

  function buildTriggerBase(slug) {
    return settings.generateBase + encodeURIComponent(slug);
  }

  function appendKey(url, requiresKey, keyValue) {
    if (!requiresKey || !keyValue) {
      return url;
    }
    const glue = url.indexOf('?') === -1 ? '?' : '&';
    return `${url}${glue}key=${encodeURIComponent(keyValue)}`;
  }

  function updateInfoUrls($card) {
    const $toggle = $card.find('.otto-db-enabled-toggle');
    const savedSlug = $.trim($toggle.data('slug') || '');
    const slugInput = $.trim($card.find('.otto-db-slug-input').val() || '');
    const slug = savedSlug || slugInput;
    if (!slug) {
      return;
    }

    const requiresKey = ($card.find('.otto-db-secret-enabled').val() || 'yes') === 'yes';
    const keyValue = $.trim($card.find('.otto-db-secret-field').val() || '');

    const downloadBase = buildDownloadBase(slug);
    const triggerBase = buildTriggerBase(slug);
    const downloadUrl = appendKey(downloadBase, requiresKey, keyValue);
    const triggerUrl = appendKey(triggerBase, requiresKey, keyValue);

    const $downloadItem = $card.find('.otto-db-info [data-role="download"]');
    const $triggerItem = $card.find('.otto-db-info [data-role="trigger"]');

    if ($downloadItem.length) {
      $downloadItem.attr('data-base', downloadBase);
      $downloadItem.attr('data-requires-key', requiresKey ? 'yes' : 'no');
      $downloadItem.find('[data-role="download-link"]').attr('href', downloadUrl);
      $downloadItem.find('[data-role="download-url"]').text(downloadUrl);
    }

    if ($triggerItem.length) {
      $triggerItem.attr('data-base', triggerBase);
      $triggerItem.attr('data-requires-key', requiresKey ? 'yes' : 'no');
      $triggerItem.find('[data-role="trigger-url"]').text(triggerUrl);
    }
  }

  function renderFieldGroup($card, postType, groupName) {
    const index = getIndex($card);
    const $column = $card.find(`.otto-db-column[data-group="${groupName}"]`).first();
    if (!$column.length) {
      return;
    }
    const $list = $column.find('[data-role="field-list"]');
    if (!$list.length) {
      return;
    }

    const existingSelections = [];
    $list.find('input[type="checkbox"]').each(function () {
      if (this.checked) {
        existingSelections.push(this.value);
      }
    });

    const dataset = (settings.ptFields[postType] && settings.ptFields[postType][groupName]) || [];
    if (!dataset.length) {
      $list.html('<p class="otto-db-empty">' + escapeHtml(settings.strings.noFields) + '</p>');
      return;
    }

    const html = dataset
      .map((item) => {
        const value = item.value || '';
        const label = item.label || item.value || '';
        const checked = existingSelections.indexOf(value) !== -1 ? ' checked' : '';
        return (
          '<label><input type="checkbox" name="tpl_fields_' +
          escapeHtml(index) +
          '[]" value="' +
          escapeHtml(value) +
          '"' +
          checked +
          ' /> ' +
          escapeHtml(label) +
          '</label><br />'
        );
      })
      .join('');

    $list.html(html);
  }

  function refreshFieldGroups($card, postType) {
    groups.forEach((group) => renderFieldGroup($card, postType, group));
  }

  function handlePostTypeChange($card) {
    const { value } = currentPostType($card);
    if (!value) {
      return;
    }
    refreshFieldGroups($card, value);
    updateHeaderTexts($card);
    updateInfoUrls($card);
  }

  function initCollapse($card) {
    const $button = $card.find('.otto-db-collapse').first();
    const $body = $card.find('.otto-db-template-body').first();
    $button.on('click', function () {
      const expanded = $(this).attr('aria-expanded') === 'true';
      $(this).attr('aria-expanded', (!expanded).toString());
      $card.toggleClass('is-collapsed', expanded);
      if (expanded) {
        $body.attr('hidden', 'hidden');
      } else {
        $body.removeAttr('hidden');
      }
    });
  }

  function initLabelWatcher($card) {
    $card.find('.otto-db-label-input').on('input', function () {
      updateHeaderTexts($card);
    });
  }

  function initSlugWatcher($card) {
    $card.find('.otto-db-slug-input').on('input', function () {
      updateHeaderTexts($card);
      updateInfoUrls($card);
    });
  }

  function initPostTypeWatcher($card) {
    $card.find('.otto-db-pt-select').on('change', function () {
      handlePostTypeChange($card);
    });
  }

  function initSecretToggle($card) {
    $card.find('.otto-db-secret-toggle').on('change', function () {
      const checked = $(this).is(':checked');
      $card.find('.otto-db-secret-enabled').val(checked ? 'yes' : 'no');
      updateInfoUrls($card);
    });

    $card.find('.otto-db-secret-field').on('input', function () {
      updateInfoUrls($card);
    });
  }

  function initGenerateKey($card) {
    $card.find('.otto-db-generate-key').on('click', function () {
      const $btn = $(this);
      if ($btn.prop('disabled')) {
        return;
      }
      $btn.prop('disabled', true).addClass('is-busy');
      $.post(settings.ajaxUrl, {
        action: 'otto_db_generate_secret',
        nonce: settings.secretNonce,
      })
        .done((response) => {
          if (response && response.success && response.data && response.data.key) {
            const $field = getCard($btn).find('.otto-db-secret-field');
            $field.val(response.data.key).trigger('input');
          } else {
            window.alert(settings.strings.secretError);
          }
        })
        .fail(() => {
          window.alert(settings.strings.secretError);
        })
        .always(() => {
          $btn.prop('disabled', false).removeClass('is-busy');
        });
    });
  }

  function initToggle($card) {
    const $toggle = $card.find('.otto-db-enabled-toggle');
    const $hidden = $card.find('.otto-db-enabled-input');
    $toggle.on('change', function () {
      const checked = $(this).is(':checked');
      const previous = $hidden.val() === 'yes';
      const savedSlug = $.trim($toggle.data('slug') || '');

      $hidden.val(checked ? 'yes' : 'no');
      $card.toggleClass('is-disabled', !checked);

      if (!savedSlug) {
        return;
      }

      $.post(settings.ajaxUrl, {
        action: 'otto_db_toggle_template',
        nonce: settings.toggleNonce,
        slug: savedSlug,
        enabled: checked ? 'yes' : 'no',
      })
        .done((response) => {
          if (!response || !response.success) {
            throw new Error('toggle_failed');
          }
        })
        .fail(() => {
          window.alert(settings.strings.toggleError);
          $toggle.prop('checked', previous);
          $hidden.val(previous ? 'yes' : 'no');
          $card.toggleClass('is-disabled', !previous);
        })
        .always(() => {
          updateInfoUrls($card);
        });
    });
  }

  function initCard($card) {
    initCollapse($card);
    initLabelWatcher($card);
    initSlugWatcher($card);
    initPostTypeWatcher($card);
    initSecretToggle($card);
    initGenerateKey($card);
    initToggle($card);

    const hiddenVal = $card.find('.otto-db-enabled-input').val() || 'yes';
    if (hiddenVal === 'no') {
      $card.addClass('is-disabled');
    }

    const pt = currentPostType($card).value || 'post';
    refreshFieldGroups($card, pt);
    updateHeaderTexts($card);
    updateInfoUrls($card);
  }

  function initialiseExistingCards() {
    $('.otto-db-template-card').each(function () {
      const $card = $(this);
      if ($card.data('initialised')) {
        return;
      }
      $card.data('initialised', true);
      initCard($card);
    });
  }

  function renumberCards() {
    const $cards = $('#otto-db-templates').find('.otto-db-template-card');
    $cards.each(function (i) {
      const $card = $(this);
      const oldIndex = $card.data('index');
      $card.attr('data-index', i).data('index', i);

      $card.find('[data-index]').each(function () {
        $(this).attr('data-index', i);
      });

      // Update collapse target and body id
      $card.find('.otto-db-template-body').attr('id', 'otto-db-template-body-' + i);
      $card.find('.otto-db-collapse').attr('data-target', 'otto-db-template-body-' + i);

      // Update input names that use index
      $card.find('input, select, textarea').each(function () {
        const $el = $(this);
        const name = $el.attr('name');
        if (!name) return;
        let newName = name;
        newName = newName.replace(/tpl_enabled\\[[^\\]]+\\]/, 'tpl_enabled[' + i + ']');
        newName = newName.replace(/tpl_secret_enabled\\[[^\\]]+\\]/, 'tpl_secret_enabled[' + i + ']');
        newName = newName.replace(/tpl_fields_[^\\[]+\\[\\]/, 'tpl_fields_' + i + '[]');
        if (newName !== name) {
          $el.attr('name', newName);
        }
      });

      // Update data attributes that include old index
      $card.find('.otto-db-enabled-toggle').attr('data-index', i);
      $card.find('.otto-db-label-input').attr('data-index', i);
      $card.find('.otto-db-slug-input').attr('data-index', i);
      $card.find('.otto-db-pt-select').attr('data-index', i);
      $card.find('.otto-db-secret-enabled').attr('data-index', i);
      $card.find('.otto-db-secret-toggle').attr('data-index', i);
      $card.find('.otto-db-secret-field').attr('data-index', i);
      $card.find('.otto-db-generate-key').attr('data-index', i);
      $card.find('.otto-db-columns').attr('data-index', i);
    });
  }

  function initRemove($card) {
    $card.find('.otto-db-remove-template').on('click', function () {
      if (!window.confirm('Remove this template? This cannot be undone.')) {
        return;
      }
      $card.remove();
      renumberCards();
    });
  }

  $('#otto-db-add-template').on('click', function () {
    const $wrap = $('#otto-db-templates');
    const idx = $wrap.find('.otto-db-template-card').length;
    const html = $('#otto-db-template-block').html().replace(/__IDX__/g, String(idx));
    const $card = $(html);
    $wrap.append($card);
    initCard($card);
    initRemove($card);
    refreshFieldGroups($card, 'post');
    renumberCards();
  });

  initialiseExistingCards();
  $('.otto-db-template-card').each(function () {
    initRemove($(this));
  });
})(jQuery);
