<?php
if (!defined('ABSPATH')) exit;

class OTTO_Data_Binder_Admin {
    private $option_key;
    private $cron_base;
    private $parent_slug;

    public function __construct($option_key, $cron_base, $parent_slug = null) {
        $this->option_key = $option_key;
        $this->cron_base  = $cron_base;
        $this->parent_slug = $parent_slug;
    }

    public function hooks() {
        add_action('admin_menu', [$this, 'menu']);
        add_action('admin_init', [$this, 'handle_post']);
        add_action('admin_enqueue_scripts', [$this, 'assets']);
        add_action('wp_ajax_otto_db_toggle_template', [$this, 'ajax_toggle_template']);
        add_action('wp_ajax_otto_db_generate_secret', [$this, 'ajax_generate_secret']);
    }

    public function menu() {
        if ($this->parent_slug) {
            add_submenu_page(
                $this->parent_slug,
                'Data Tool',
                'Data Tool',
                'manage_options',
                OTTO_DB_SLUG,
                [$this, 'render']
            );
        } else {
            add_menu_page(
                'OTTO Data Binder',
                'OTTO Data Binder',
                'manage_options',
                OTTO_DB_SLUG,
                [$this, 'render'],
                'dashicons-media-spreadsheet',
                82
            );
        }
    }

    public function assets($hook) {
        $expected = $this->parent_slug ? ($this->parent_slug . '_page_' . OTTO_DB_SLUG) : ('toplevel_page_' . OTTO_DB_SLUG);
        if ($hook !== $expected && strpos($hook, OTTO_DB_SLUG) === false) {
            return;
        }
        wp_enqueue_style('otto-db-admin', OTTO_DB_URL . 'admin.css', [], OTTO_DB_VERSION);
        wp_enqueue_script('otto-db-admin', OTTO_DB_URL . 'admin.js', ['jquery'], OTTO_DB_VERSION, true);
        wp_localize_script('otto-db-admin', 'OTTO_DB', [
            'nonce'         => wp_create_nonce('otto_db_nonce'),
            'ajaxUrl'       => admin_url('admin-ajax.php'),
            'toggleNonce'   => wp_create_nonce('otto_db_toggle'),
            'secretNonce'   => wp_create_nonce('otto_db_generate_secret'),
            'ptFields'      => $this->pt_fields_map(), // for quick swaps client-side
            'fileBase'      => home_url('/wp-json/otto-data-binder/v1/file/'),
            'generateBase'  => home_url('/wp-json/otto-data-binder/v1/generate/'),
            'strings'       => [
                'untitled'   => __('Untitled binder', 'otto-data-binder'),
                'unsaved'    => __('Not yet saved', 'otto-data-binder'),
                'toggleError'=> __('Unable to update template.', 'otto-data-binder'),
                'secretError'=> __('Unable to generate a new key.', 'otto-data-binder'),
                'noFields'   => __('No fields detected for this post type yet.', 'otto-data-binder'),
            ],
        ]);
    }

    private function pt_fields_map() {
        $out = [];
        $pt = 'product';
        $obj = get_post_type_object('product');
        if ($obj) {
            $core = [];
            foreach (otto_db_core_fields_for($pt) as $field_key) {
                $core[] = ['value' => $field_key, 'label' => $field_key];
            }

            $meta = [];
            foreach (otto_db_detect_meta_keys_for($pt) as $field_key) {
                $meta[] = ['value' => $field_key, 'label' => $field_key];
            }

            $acf = [];
            foreach (otto_db_acf_fields_for($pt) as $key => $label) {
                $acf[] = ['value' => $key, 'label' => $label];
            }

            $tax = [];
            foreach (otto_db_taxonomies_for($pt) as $slug => $label) {
                $tax[] = ['value' => 'tax:' . $slug, 'label' => $label];
            }

            $out[$pt] = [
                'core' => $core,
                'meta' => $meta,
                'acf'  => $acf,
                'tax'  => $tax,
            ];
        }
        return $out;
    }

    public function handle_post() {
        if (!current_user_can('manage_options')) return;
        if (!isset($_POST['otto_db_save']) || !check_admin_referer('otto_db_save', 'otto_db_nonce')) return;

        $settings = otto_db_get_settings($this->option_key);
        $settings['cleanup_on_uninstall']        = isset($_POST['cleanup_on_uninstall']) ? 'yes' : 'no';
        $settings['delete_binders_on_uninstall'] = isset($_POST['delete_binders_on_uninstall']) ? 'yes' : 'no';

        // Read templates from POST arrays
        $tpl_enabled = isset($_POST['tpl_enabled']) ? array_map('sanitize_text_field', (array)$_POST['tpl_enabled']) : [];
        $tpl_slugs   = isset($_POST['tpl_slug']) ? array_map('sanitize_title_with_dashes', (array)$_POST['tpl_slug']) : [];
        $tpl_labels  = isset($_POST['tpl_label']) ? array_map('sanitize_text_field', (array)$_POST['tpl_label']) : [];
        $tpl_pts     = isset($_POST['tpl_post_type']) ? array_map('sanitize_text_field', (array)$_POST['tpl_post_type']) : [];
        $tpl_sched   = isset($_POST['tpl_schedule']) ? array_map('sanitize_text_field', (array)$_POST['tpl_schedule']) : [];
        $tpl_secretE = isset($_POST['tpl_secret_enabled']) ? array_map('sanitize_text_field', (array)$_POST['tpl_secret_enabled']) : [];
        $tpl_secretK = isset($_POST['tpl_secret_key']) ? array_map('sanitize_text_field', (array)$_POST['tpl_secret_key']) : [];

        $tpl_status  = isset($_POST['tpl_post_status']) ? (array)$_POST['tpl_post_status'] : [];
        $tpl_stock   = isset($_POST['tpl_only_in_stock']) ? (array)$_POST['tpl_only_in_stock'] : [];
        $tpl_wl      = isset($_POST['tpl_sku_whitelist']) ? array_map('sanitize_textarea_field', (array)$_POST['tpl_sku_whitelist']) : [];
        $tpl_bl      = isset($_POST['tpl_sku_blacklist']) ? array_map('sanitize_textarea_field', (array)$_POST['tpl_sku_blacklist']) : [];
        $tpl_cat     = isset($_POST['tpl_category_slugs']) ? array_map('sanitize_textarea_field', (array)$_POST['tpl_category_slugs']) : [];
        $tpl_tag     = isset($_POST['tpl_tag_slugs']) ? array_map('sanitize_textarea_field', (array)$_POST['tpl_tag_slugs']) : [];
        $tpl_mkey    = isset($_POST['tpl_meta_key']) ? array_map('sanitize_text_field', (array)$_POST['tpl_meta_key']) : [];
        $tpl_mop     = isset($_POST['tpl_meta_op']) ? array_map('sanitize_text_field', (array)$_POST['tpl_meta_op']) : [];
        $tpl_mval    = isset($_POST['tpl_meta_value']) ? array_map('sanitize_textarea_field', (array)$_POST['tpl_meta_value']) : [];

        $templates = [];
        $count = count($tpl_slugs);
        for ($i=0; $i<$count; $i++) {
            $slug = $tpl_slugs[$i];
            if ($slug === '') continue;

            $tpl = otto_db_template_defaults();
            $tpl['enabled']    = (isset($tpl_enabled[$i]) && $tpl_enabled[$i] === 'no') ? 'no' : 'yes';
            $tpl['slug']       = $slug;
            $tpl['label']      = $tpl_labels[$i] ?? $slug;
            $tpl['post_type']  = 'product';
            $tpl['schedule']   = in_array(($tpl_sched[$i] ?? 'manual'), ['manual','hourly','twicedaily','daily'], true) ? $tpl_sched[$i] : 'manual';
            $tpl['secret_enabled'] = (isset($tpl_secretE[$i]) && $tpl_secretE[$i] === 'yes') ? 'yes' : 'no';
            $tpl['secret_key'] = $tpl_secretK[$i] ?? '';

            $tpl['filters']['post_status']    = (isset($tpl_status[$i]) && $tpl_status[$i] === 'any') ? 'any' : 'publish';
            $tpl['filters']['only_in_stock']  = (isset($tpl_stock[$i]) && $tpl_stock[$i] === 'yes') ? 'yes' : 'no';
            $tpl['filters']['sku_whitelist']  = $tpl_wl[$i] ?? '';
            $tpl['filters']['sku_blacklist']  = $tpl_bl[$i] ?? '';
            $tpl['filters']['category_slugs'] = $tpl_cat[$i] ?? '';
            $tpl['filters']['tag_slugs']      = $tpl_tag[$i] ?? '';
            $tpl['filters']['meta_key']       = $tpl_mkey[$i] ?? '';
            $tpl['filters']['meta_op']        = in_array(($tpl_mop[$i] ?? 'equals'), ['equals','not_equals','contains','gt','gte','lt','lte'], true) ? $tpl_mop[$i] : 'equals';
            $tpl['filters']['meta_value']     = $tpl_mval[$i] ?? '';

            // Fields for this template come as merged arrays: core/meta/acf/tax prefixes
            $field_key = 'tpl_fields_' . $i;
            $tpl['fields'] = isset($_POST[$field_key]) ? array_values(array_unique(array_map('sanitize_text_field', (array)$_POST[$field_key]))) : [];

            $templates[] = $tpl;
        }

        $settings['templates'] = $templates;

        otto_db_ensure_output_dir($settings['output_dir']);
        otto_db_save_settings($this->option_key, $settings);

        // Reschedule crons
        otto_db_reschedule_all_templates($this->cron_base, $templates);

        add_settings_error('otto_db', 'settings_saved', 'Settings saved.', 'updated');

        // Generate now if asked (per-template buttons handled via GET param below in render())
        if (isset($_POST['otto_db_generate_now_all'])) {
            foreach ($templates as $tpl) {
                $hook = otto_db_event_name($this->cron_base, $tpl['slug']);
                do_action($hook, $tpl['slug']);
            }
            add_settings_error('otto_db', 'generated_now', 'All templates generated.', 'updated');
        }
    }

    public function render() {
        $settings = otto_db_get_settings($this->option_key);
        // Force product exports only
        $productType = get_post_type_object('product');
        $post_types = $productType ? ['product' => $productType] : [];

        // Per-template Generate Now action via ?otto_db_gen={slug}
        if (isset($_GET['otto_db_gen']) && current_user_can('manage_options')) {
            $slug = sanitize_title_with_dashes($_GET['otto_db_gen']);
            $hook = otto_db_event_name($this->cron_base, $slug);
            do_action($hook, $slug);
            echo '<div class="updated notice is-dismissible"><p>Generated: ' . esc_html($slug) . '</p></div>';
        }

        settings_errors('otto_db');
        ?>
        <div class="wrap">
            <!-- About / Branding Card -->
            <div class="otto-db-about">
                <div class="otto-db-about-right">
                    <h1 class="otto-db-about-title">Suggent Data Tool</h1>
                    <p class="otto-db-about-sub">Export WooCommerce product data to JSON for Suggent bots.</p>
                    <p class="otto-db-about-desc">Select product fields, then share the JSON URL.</p>
                </div>
            </div>

            <form method="post" class="otto-db-form">
                <?php wp_nonce_field('otto_db_save', 'otto_db_nonce'); ?>

                <h2 class="title">Product Export Templates</h2>
                <?php if (!$post_types): ?>
                    <div class="notice notice-error"><p>WooCommerce is not active. Install/activate WooCommerce to export products.</p></div>
                <?php endif; ?>
                <div id="otto-db-templates">
                    <?php
                    $i = 0;
                    if (empty($settings['templates'])) {
                        $settings['templates'][] = otto_db_template_defaults();
                    }
                    foreach ($settings['templates'] as $tpl):
                        $slug    = $tpl['slug'];
                        $label   = $tpl['label'];
                        $pt      = 'product';
                        $fields  = $tpl['fields'];
                        $filt    = $tpl['filters'];
                        $sched   = $tpl['schedule'];
                        $seEn    = $tpl['secret_enabled'];
                        $seKy    = $tpl['secret_key'];
                        $last    = $tpl['last_run'];
                        $enabled = ($tpl['enabled'] ?? 'yes') === 'no' ? 'no' : 'yes';

                        $title    = $label !== '' ? $label : ($slug !== '' ? $slug : __('Untitled binder', 'otto-data-binder'));
                        $subtitle = $slug !== '' ? $slug . ' · products' : __('Not yet saved', 'otto-data-binder');
                        $card_cls = 'otto-db-template-card';
                        if ($enabled === 'no') $card_cls .= ' is-disabled';

                        $download_url = '';
                        $generate_url = '';
                        $download_base = '';
                        $generate_base = '';
                        $file_size    = __('Not generated yet', 'otto-data-binder');
                        $last_time    = '—';

                        if (!empty($last['time'])) {
                            $last_time = esc_html(date_i18n(get_option('date_format') . ' ' . get_option('time_format'), intval($last['time'])));
                        }

                        if ($slug !== '') {
                            $download_base = home_url('/wp-json/otto-data-binder/v1/file/' . rawurlencode($slug) . '.json');
                            $generate_base = home_url('/wp-json/otto-data-binder/v1/generate/' . rawurlencode($slug));
                            $download_url  = $download_base;
                            $generate_url  = $generate_base;
                            if ($seEn === 'yes' && $seKy) {
                                $download_url = add_query_arg(['key' => $seKy], $download_url);
                                $generate_url = add_query_arg(['key' => $seKy], $generate_url);
                            }

                            if (!empty($last['bytes'])) {
                                $file_size = otto_db_human_bytes(intval($last['bytes']));
                            } elseif (!empty($last['time'])) {
                                $file_size = '0 B';
                            }
                        }
                    ?>
                    <div class="<?php echo esc_attr($card_cls); ?>" data-index="<?php echo esc_attr($i); ?>">
                        <input type="hidden" class="otto-db-enabled-input" name="tpl_enabled[<?php echo esc_attr($i); ?>]" value="<?php echo esc_attr($enabled); ?>" />

                        <div class="otto-db-template-header">
                            <label class="otto-db-switch">
                                <input type="checkbox"
                                       class="otto-db-enabled-toggle"
                                       data-index="<?php echo esc_attr($i); ?>"
                                       data-slug="<?php echo esc_attr($slug); ?>"
                                       <?php checked($enabled === 'yes'); ?> />
                                <span class="otto-db-switch-slider" aria-hidden="true"></span>
                                <span class="screen-reader-text"><?php esc_html_e('Enable template', 'otto-data-binder'); ?></span>
                            </label>

                            <button type="button"
                                    class="otto-db-collapse"
                                    aria-expanded="true"
                                    data-target="otto-db-template-body-<?php echo esc_attr($i); ?>"
                                    data-default-title="<?php esc_attr_e('Untitled binder', 'otto-data-binder'); ?>"
                                    data-default-sub="<?php esc_attr_e('Not yet saved', 'otto-data-binder'); ?>">
                                <span class="otto-db-template-title" data-role="template-title"><?php echo esc_html($title); ?></span>
                                <span class="otto-db-template-sub" data-role="template-sub"><?php echo esc_html($subtitle); ?></span>
                            </button>
                        </div>

                        <div class="otto-db-template-body" id="otto-db-template-body-<?php echo esc_attr($i); ?>">
                            <div class="otto-db-field-row">
                                <div class="otto-db-field otto-db-field--full">
                                    <label>Label</label>
                                    <input type="text" name="tpl_label[]" value="<?php echo esc_attr($label); ?>" placeholder="Product Price Binder" class="otto-db-label-input" data-index="<?php echo esc_attr($i); ?>" />
                                </div>
                            </div>

                            <div class="otto-db-field-row">
                                <div class="otto-db-field">
                                    <label>Slug (file name base)</label>
                                    <input type="text" name="tpl_slug[]" value="<?php echo esc_attr($slug); ?>" placeholder="e.g. product-price" required class="otto-db-slug-input" data-index="<?php echo esc_attr($i); ?>" />
                                </div>
                                <div class="otto-db-field">
                                    <label>Post type</label>
                                    <select name="tpl_post_type[]" class="otto-db-pt-select" data-index="<?php echo esc_attr($i); ?>">
                                        <option value="product" selected>Products (WooCommerce)</option>
                                    </select>
                                </div>
                                <div class="otto-db-field">
                                    <label>Schedule</label>
                                    <select name="tpl_schedule[]">
                                        <option value="manual" <?php selected($sched, 'manual'); ?>>Manual</option>
                                        <option value="hourly" <?php selected($sched, 'hourly'); ?>>Hourly</option>
                                        <option value="twicedaily" <?php selected($sched, 'twicedaily'); ?>>Twice daily</option>
                                        <option value="daily" <?php selected($sched, 'daily'); ?>>Daily</option>
                                    </select>
                                </div>
                            </div>

                            <div class="otto-db-field-row otto-db-field-row--secret">
                                <input type="hidden" name="tpl_secret_enabled[<?php echo esc_attr($i); ?>]" value="<?php echo esc_attr($seEn); ?>" class="otto-db-secret-enabled" data-index="<?php echo esc_attr($i); ?>" />
                                <div class="otto-db-field otto-db-field--shrink">
                                    <label class="otto-db-checkbox">
                                        <input type="checkbox" class="otto-db-secret-toggle" data-index="<?php echo esc_attr($i); ?>" <?php checked($seEn === 'yes'); ?> />
                                        <span>Require key</span>
                                    </label>
                                </div>
                                <div class="otto-db-field otto-db-field--grow">
                                    <label>Secret key</label>
                                    <div class="otto-db-secret-controls">
                                        <input type="text" name="tpl_secret_key[]" value="<?php echo esc_attr($seKy); ?>" class="wide otto-db-secret-field" data-index="<?php echo esc_attr($i); ?>" />
                                        <button type="button" class="button otto-db-generate-key" data-index="<?php echo esc_attr($i); ?>">Generate key</button>
                                    </div>
                                </div>
                            </div>

                            <div class="otto-db-columns" data-index="<?php echo esc_attr($i); ?>">
                                <div class="otto-db-column" data-group="core">
                                    <h3>Core fields</h3>
                                    <div class="otto-db-scroll" data-role="field-list">
                                        <?php foreach (otto_db_core_fields_for($pt) as $f): ?>
                                            <label><input type="checkbox" name="tpl_fields_<?php echo esc_attr($i); ?>[]" value="<?php echo esc_attr($f); ?>" <?php checked(in_array($f, $fields, true)); ?> /> <?php echo esc_html($f); ?></label><br />
                                        <?php endforeach; ?>
                                    </div>
                                </div>
                                <div class="otto-db-column" data-group="meta">
                                    <h3>Custom/meta fields</h3>
                                    <div class="otto-db-scroll" data-role="field-list">
                                        <?php foreach (otto_db_detect_meta_keys_for($pt) as $f): ?>
                                            <label><input type="checkbox" name="tpl_fields_<?php echo esc_attr($i); ?>[]" value="<?php echo esc_attr($f); ?>" <?php checked(in_array($f, $fields, true)); ?> /> <?php echo esc_html($f); ?></label><br />
                                        <?php endforeach; ?>
                                    </div>
                                </div>
                                <div class="otto-db-column" data-group="acf">
                                    <h3>ACF fields</h3>
                                    <div class="otto-db-scroll" data-role="field-list">
                                        <?php foreach (otto_db_acf_fields_for($pt) as $k => $lbl): ?>
                                            <label><input type="checkbox" name="tpl_fields_<?php echo esc_attr($i); ?>[]" value="<?php echo esc_attr($k); ?>" <?php checked(in_array($k, $fields, true)); ?> /> <?php echo esc_html($lbl); ?></label><br />
                                        <?php endforeach; ?>
                                    </div>
                                </div>
                                <div class="otto-db-column" data-group="tax">
                                    <h3>Taxonomies</h3>
                                    <div class="otto-db-scroll" data-role="field-list">
                                        <?php foreach (otto_db_taxonomies_for($pt) as $tax => $lbl): $val = 'tax:' . $tax; ?>
                                            <label><input type="checkbox" name="tpl_fields_<?php echo esc_attr($i); ?>[]" value="<?php echo esc_attr($val); ?>" <?php checked(in_array($val, $fields, true)); ?> /> <?php echo esc_html($lbl); ?></label><br />
                                        <?php endforeach; ?>
                                    </div>
                                </div>
                            </div>

                            <h3>Output filters</h3>
                            <div class="otto-db-field-row">
                                <div class="otto-db-field">
                                    <label>Post status</label>
                                    <select name="tpl_post_status[]">
                                        <option value="publish" <?php selected($filt['post_status'], 'publish'); ?>>Published only</option>
                                        <option value="any" <?php selected($filt['post_status'], 'any'); ?>>All statuses</option>
                                    </select>
                                </div>
                                <div class="otto-db-field">
                                    <label>Woo in-stock only</label>
                                    <select name="tpl_only_in_stock[]">
                                        <option value="no" <?php selected($filt['only_in_stock'], 'no'); ?>>No</option>
                                        <option value="yes" <?php selected($filt['only_in_stock'], 'yes'); ?>>Yes</option>
                                    </select>
                                </div>
                                <div class="otto-db-field">
                                    <label>SKU whitelist (CSV)</label>
                                    <input type="text" name="tpl_sku_whitelist[]" value="<?php echo esc_attr($filt['sku_whitelist']); ?>" />
                                </div>
                                <div class="otto-db-field">
                                    <label>SKU blacklist (CSV)</label>
                                    <input type="text" name="tpl_sku_blacklist[]" value="<?php echo esc_attr($filt['sku_blacklist']); ?>" />
                                </div>
                            </div>

                            <div class="otto-db-field-row">
                                <div class="otto-db-field">
                                    <label>Category slugs (CSV)</label>
                                    <input type="text" name="tpl_category_slugs[]" value="<?php echo esc_attr($filt['category_slugs'] ?? ''); ?>" placeholder="e.g. mattresses, bedding" />
                                </div>
                                <div class="otto-db-field">
                                    <label>Tag slugs (CSV)</label>
                                    <input type="text" name="tpl_tag_slugs[]" value="<?php echo esc_attr($filt['tag_slugs'] ?? ''); ?>" placeholder="e.g. cooling, hybrid" />
                                </div>
                                <div class="otto-db-field">
                                    <label>Meta filter key</label>
                                    <input type="text" name="tpl_meta_key[]" value="<?php echo esc_attr($filt['meta_key'] ?? ''); ?>" placeholder="e.g. _price" />
                                </div>
                                <div class="otto-db-field">
                                    <label>Meta filter</label>
                                    <select name="tpl_meta_op[]">
                                        <option value="equals" <?php selected(($filt['meta_op'] ?? 'equals'), 'equals'); ?>>Equals</option>
                                        <option value="not_equals" <?php selected(($filt['meta_op'] ?? 'equals'), 'not_equals'); ?>>Not equals</option>
                                        <option value="contains" <?php selected(($filt['meta_op'] ?? 'equals'), 'contains'); ?>>Contains</option>
                                        <option value="gt" <?php selected(($filt['meta_op'] ?? 'equals'), 'gt'); ?>>Greater than</option>
                                        <option value="gte" <?php selected(($filt['meta_op'] ?? 'equals'), 'gte'); ?>>Greater or equal</option>
                                        <option value="lt" <?php selected(($filt['meta_op'] ?? 'equals'), 'lt'); ?>>Less than</option>
                                        <option value="lte" <?php selected(($filt['meta_op'] ?? 'equals'), 'lte'); ?>>Less or equal</option>
                                    </select>
                                </div>
                                <div class="otto-db-field">
                                    <label>Meta filter value</label>
                                    <input type="text" name="tpl_meta_value[]" value="<?php echo esc_attr($filt['meta_value'] ?? ''); ?>" placeholder="e.g. 499" />
                                </div>
                            </div>

                            <div class="otto-db-info">
                                <h3>Useful information</h3>
                                <?php if ($slug === ''): ?>
                                    <p>Save the template to unlock export links and file metadata.</p>
                                <?php else: ?>
                                    <ul>
                                        <li class="otto-db-info-row" data-role="download" data-base="<?php echo esc_attr($download_base); ?>" data-requires-key="<?php echo $seEn === 'yes' ? 'yes' : 'no'; ?>">
                                            <strong>Download JSON:</strong>
                                            <a href="<?php echo esc_url($download_url); ?>" target="_blank" rel="noopener" data-role="download-link">Open link</a><br />
                                            <code data-role="download-url"><?php echo esc_html($download_url); ?></code>
                                        </li>
                                        <li class="otto-db-info-row" data-role="trigger" data-base="<?php echo esc_attr($generate_base); ?>" data-requires-key="<?php echo $seEn === 'yes' ? 'yes' : 'no'; ?>">
                                            <strong>Trigger URL:</strong>
                                            <code data-role="trigger-url"><?php echo esc_html($generate_url); ?></code>
                                        </li>
                                        <li><strong>File size:</strong> <?php echo esc_html($file_size); ?></li>
                                        <li><strong>Last generated:</strong> <?php echo esc_html($last_time); ?></li>
                                    </ul>
                                <?php endif; ?>
                            </div>

                            <div class="otto-db-template-actions">
                                <?php if ($slug !== ''): ?>
                                    <a class="button" href="<?php echo esc_url(add_query_arg(['page' => OTTO_DB_SLUG, 'otto_db_gen' => $slug])); ?>">Generate now</a>
                                <?php else: ?>
                                    <button type="button" class="button" disabled>Generate now</button>
                                <?php endif; ?>
                                <span class="description">Remember to save changes after editing a template.</span>
                                <button type="button" class="button otto-db-remove-template">Remove template</button>
                            </div>
                        </div>
                    </div>
                    <?php $i++; endforeach; ?>
                </div>

                <p><button type="button" class="button" id="otto-db-add-template">Add template</button></p>

                <hr />
                <h2 class="title">Uninstall cleanup</h2>
                <label><input type="checkbox" name="cleanup_on_uninstall" <?php checked($settings['cleanup_on_uninstall']==='yes'); ?> /> Remove settings on uninstall</label><br/>
                <label><input type="checkbox" name="delete_binders_on_uninstall" <?php checked($settings['delete_binders_on_uninstall']==='yes'); ?> /> Delete generated binder files on uninstall</label>

                <p class="submit">
                    <button type="submit" name="otto_db_save" class="button button-primary">Save</button>
                    <button type="submit" name="otto_db_generate_now_all" class="button">Generate all now</button>
                </p>
            </form>

            <!-- Hidden template for JS cloning -->
            <template id="otto-db-template-block">
                <?php
                $i         = '__IDX__';
                $pt        = 'post';
                $tpl_def   = otto_db_template_defaults();
                $filt      = $tpl_def['filters'];
                $enabled   = 'yes';
                ?>
                <div class="otto-db-template-card" data-index="<?php echo esc_attr($i); ?>">
                    <input type="hidden" class="otto-db-enabled-input" name="tpl_enabled[<?php echo esc_attr($i); ?>]" value="<?php echo esc_attr($enabled); ?>" />

                    <div class="otto-db-template-header">
                        <label class="otto-db-switch">
                            <input type="checkbox" class="otto-db-enabled-toggle" data-index="<?php echo esc_attr($i); ?>" data-slug="" checked />
                            <span class="otto-db-switch-slider" aria-hidden="true"></span>
                            <span class="screen-reader-text"><?php esc_html_e('Enable template', 'otto-data-binder'); ?></span>
                        </label>
                        <button type="button" class="otto-db-collapse" aria-expanded="true" data-target="otto-db-template-body-<?php echo esc_attr($i); ?>" data-default-title="<?php esc_attr_e('Untitled binder', 'otto-data-binder'); ?>" data-default-sub="<?php esc_attr_e('Not yet saved', 'otto-data-binder'); ?>">
                            <span class="otto-db-template-title" data-role="template-title"><?php esc_html_e('Untitled binder', 'otto-data-binder'); ?></span>
                            <span class="otto-db-template-sub" data-role="template-sub"><?php esc_html_e('Not yet saved', 'otto-data-binder'); ?></span>
                        </button>
                    </div>

                    <div class="otto-db-template-body" id="otto-db-template-body-<?php echo esc_attr($i); ?>">
                        <div class="otto-db-field-row">
                            <div class="otto-db-field otto-db-field--full">
                                <label>Label</label>
                                <input type="text" name="tpl_label[]" value="" placeholder="Product Info Binder" class="otto-db-label-input" data-index="<?php echo esc_attr($i); ?>" />
                            </div>
                        </div>

                        <div class="otto-db-field-row">
                            <div class="otto-db-field">
                                <label>Slug (file name base)</label>
                                <input type="text" name="tpl_slug[]" value="" placeholder="e.g. product-info" required class="otto-db-slug-input" data-index="<?php echo esc_attr($i); ?>" />
                            </div>
                            <div class="otto-db-field">
                                <label>Post type</label>
                                <select name="tpl_post_type[]" class="otto-db-pt-select" data-index="<?php echo esc_attr($i); ?>">
                                    <?php foreach ($post_types as $ppt => $obj): ?>
                                        <option value="<?php echo esc_attr($ppt); ?>"><?php echo esc_html($obj->labels->name . ' (' . $ppt . ')'); ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div class="otto-db-field">
                                <label>Schedule</label>
                                <select name="tpl_schedule[]">
                                    <option value="manual">Manual</option>
                                    <option value="hourly">Hourly</option>
                                    <option value="twicedaily">Twice daily</option>
                                    <option value="daily">Daily</option>
                                </select>
                            </div>
                        </div>

                        <div class="otto-db-field-row otto-db-field-row--secret">
                            <input type="hidden" name="tpl_secret_enabled[<?php echo esc_attr($i); ?>]" value="yes" class="otto-db-secret-enabled" data-index="<?php echo esc_attr($i); ?>" />
                            <div class="otto-db-field otto-db-field--shrink">
                                <label class="otto-db-checkbox">
                                    <input type="checkbox" class="otto-db-secret-toggle" data-index="<?php echo esc_attr($i); ?>" checked />
                                    <span>Require key</span>
                                </label>
                            </div>
                            <div class="otto-db-field otto-db-field--grow">
                                <label>Secret key</label>
                                <div class="otto-db-secret-controls">
                                    <input type="text" name="tpl_secret_key[]" value="" class="wide otto-db-secret-field" data-index="<?php echo esc_attr($i); ?>" />
                                    <button type="button" class="button otto-db-generate-key" data-index="<?php echo esc_attr($i); ?>">Generate key</button>
                                </div>
                            </div>
                        </div>

                            <div class="otto-db-columns" data-index="<?php echo esc_attr($i); ?>">
                                <div class="otto-db-column" data-group="core">
                                    <h3>Core fields</h3>
                                <div class="otto-db-scroll" data-role="field-list">
                                    <?php foreach (otto_db_core_fields_for($pt) as $f): ?>
                                        <label><input type="checkbox" name="tpl_fields_<?php echo esc_attr($i); ?>[]" value="<?php echo esc_attr($f); ?>" /> <?php echo esc_html($f); ?></label><br />
                                    <?php endforeach; ?>
                                </div>
                            </div>
                            <div class="otto-db-column" data-group="meta">
                                <h3>Custom/meta fields</h3>
                                <div class="otto-db-scroll" data-role="field-list"></div>
                            </div>
                            <div class="otto-db-column" data-group="acf">
                                <h3>ACF fields</h3>
                                <div class="otto-db-scroll" data-role="field-list"></div>
                            </div>
                            <div class="otto-db-column" data-group="tax">
                                <h3>Taxonomies</h3>
                                <div class="otto-db-scroll" data-role="field-list"></div>
                                </div>
                            </div>

                        <h3>Output filters</h3>
                        <div class="otto-db-field-row">
                            <div class="otto-db-field">
                                <label>Post status</label>
                                <select name="tpl_post_status[]">
                                    <option value="publish">Published only</option>
                                    <option value="any">All statuses</option>
                                </select>
                            </div>
                            <div class="otto-db-field">
                                <label>Woo in-stock only</label>
                                <select name="tpl_only_in_stock[]">
                                    <option value="no">No</option>
                                    <option value="yes">Yes</option>
                                </select>
                            </div>
                            <div class="otto-db-field">
                                <label>SKU whitelist (CSV)</label>
                                <input type="text" name="tpl_sku_whitelist[]" value="" />
                            </div>
                            <div class="otto-db-field">
                                <label>SKU blacklist (CSV)</label>
                                <input type="text" name="tpl_sku_blacklist[]" value="" />
                            </div>
                        </div>

                        <div class="otto-db-field-row">
                            <div class="otto-db-field">
                                <label>Category slugs (CSV)</label>
                                <input type="text" name="tpl_category_slugs[]" value="" placeholder="e.g. mattresses, bedding" />
                            </div>
                            <div class="otto-db-field">
                                <label>Tag slugs (CSV)</label>
                                <input type="text" name="tpl_tag_slugs[]" value="" placeholder="e.g. cooling, hybrid" />
                            </div>
                            <div class="otto-db-field">
                                <label>Meta filter key</label>
                                <input type="text" name="tpl_meta_key[]" value="" placeholder="e.g. _price" />
                            </div>
                            <div class="otto-db-field">
                                <label>Meta filter</label>
                                <select name="tpl_meta_op[]">
                                    <option value="equals">Equals</option>
                                    <option value="not_equals">Not equals</option>
                                    <option value="contains">Contains</option>
                                    <option value="gt">Greater than</option>
                                    <option value="gte">Greater or equal</option>
                                    <option value="lt">Less than</option>
                                    <option value="lte">Less or equal</option>
                                </select>
                            </div>
                            <div class="otto-db-field">
                                <label>Meta filter value</label>
                                <input type="text" name="tpl_meta_value[]" value="" placeholder="e.g. 499" />
                            </div>
                        </div>

                        <div class="otto-db-info">
                            <h3>Useful information</h3>
                            <p>Save the template to unlock export links and file metadata.</p>
                        </div>

                        <div class="otto-db-template-actions">
                            <span class="description">Remember to save changes after editing a template.</span>
                            <button type="button" class="button otto-db-remove-template">Remove template</button>
                        </div>
                    </div>
                </div>
            </template>
        </div>
        <?php
    }

    public function ajax_toggle_template() {
        check_ajax_referer('otto_db_toggle', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Not allowed', 'otto-data-binder')], 403);
        }

        $slug    = isset($_POST['slug']) ? sanitize_title_with_dashes(wp_unslash($_POST['slug'])) : '';
        $enabled = (isset($_POST['enabled']) && wp_unslash($_POST['enabled']) === 'yes') ? 'yes' : 'no';

        if ($slug === '') {
            wp_send_json_error(['message' => __('Missing template slug.', 'otto-data-binder')], 400);
        }

        $settings = otto_db_get_settings($this->option_key);
        $found    = false;

        foreach ($settings['templates'] as &$tpl) {
            if ($tpl['slug'] === $slug) {
                $tpl['enabled'] = $enabled;
                $found = true;
                break;
            }
        }
        unset($tpl);

        if (!$found) {
            wp_send_json_error(['message' => __('Template not found.', 'otto-data-binder')], 404);
        }

        otto_db_save_settings($this->option_key, $settings);
        otto_db_reschedule_all_templates($this->cron_base, $settings['templates']);

        wp_send_json_success(['enabled' => $enabled]);
    }

    public function ajax_generate_secret() {
        check_ajax_referer('otto_db_generate_secret', 'nonce');
        if (!current_user_can('manage_options')) {
            wp_send_json_error(['message' => __('Not allowed', 'otto-data-binder')], 403);
        }

        $key = wp_generate_password(24, false, false);
        wp_send_json_success(['key' => $key]);
    }
}
