<?php
if (!defined('ABSPATH')) exit;

class OTTO_Data_Binder_Generator {
    private $option_key;
    private $cron_base;

    public function __construct($option_key, $cron_base) {
        $this->option_key = $option_key;
        $this->cron_base  = $cron_base;
    }

    public function hooks() {
        // We’ll dynamically add actions for each template on admin save; also add a catch-all.
        add_action('init', [$this, 'attach_dynamic_cron_hooks']);
    }

    public function attach_dynamic_cron_hooks() {
        $settings = otto_db_get_settings($this->option_key);
        foreach ($settings['templates'] as $tpl) {
            if (empty($tpl['slug']) || ($tpl['enabled'] ?? 'yes') === 'no') continue;
            $hook = otto_db_event_name($this->cron_base, $tpl['slug']);
            add_action($hook, [$this, 'generate_by_slug'], 10, 1);
        }
    }

    /** Public callable to generate one template by slug */
    public function generate_by_slug($slug) {
        $settings = otto_db_get_settings($this->option_key);
        $tpl = null; $idx = -1;
        foreach ($settings['templates'] as $i=>$t) {
            if ($t['slug'] === $slug) { $tpl = $t; $idx = $i; break; }
        }
        if (!$tpl || ($tpl['enabled'] ?? 'yes') === 'no') return false;

        $data = $this->build_data($tpl);
        $path = $this->write_json($settings['output_dir'], $tpl['slug'], $tpl['post_type'], $data);

        // Update last_run
        $settings['templates'][$idx]['last_run'] = [
            'time'  => time(),
            'bytes' => (file_exists($path) ? filesize($path) : 0),
            'path'  => $path,
        ];
        otto_db_save_settings($this->option_key, $settings);
        return true;
    }

    private function build_data($tpl) {
        $pt   = $tpl['post_type'];
        $filt = $tpl['filters'];
        $status = ($filt['post_status'] === 'any') ? 'any' : 'publish';
        $ids = get_posts([
            'post_type'      => $pt,
            'post_status'    => $status,
            'posts_per_page' => -1,
            'fields'         => 'ids',
            'no_found_rows'  => true,
        ]);
        if (empty($ids)) return ['post_type'=>$pt,'generated'=>current_time('mysql',true),'count'=>0,'items'=>[]];

        // Woo filtering
        if ($pt === 'product') {
            $ids = $this->filter_products($ids, $filt);
        }

        $items = [];
        foreach ($ids as $id) {
            $items[] = $this->collect_record($id, $pt, $tpl['fields']);
        }

        return [
            'post_type' => $pt,
            'generated' => current_time('mysql', true),
            'count'     => count($items),
            'items'     => $items,
        ];
    }

    private function filter_products($ids, $filt) {
        $wl = otto_db_csv_to_array($filt['sku_whitelist'] ?? '');
        $bl = otto_db_csv_to_array($filt['sku_blacklist'] ?? '');
        $cats = otto_db_csv_to_array($filt['category_slugs'] ?? '');
        $tags = otto_db_csv_to_array($filt['tag_slugs'] ?? '');
        $metaKey = trim((string)($filt['meta_key'] ?? ''));
        $metaOp = (string)($filt['meta_op'] ?? 'equals');
        $metaVal = (string)($filt['meta_value'] ?? '');
        $out = [];
        foreach ($ids as $id) {
            $sku = get_post_meta($id, '_sku', true);
            if (!empty($wl) && !in_array((string)$sku, $wl, true)) continue;
            if (!empty($bl) && in_array((string)$sku, $bl, true)) continue;
            if (($filt['only_in_stock'] ?? 'no') === 'yes') {
                $st = get_post_meta($id, '_stock_status', true);
                if ($st !== 'instock') continue;
            }
            if (!empty($cats) && !has_term($cats, 'product_cat', $id)) continue;
            if (!empty($tags) && !has_term($tags, 'product_tag', $id)) continue;
            if ($metaKey !== '' && $metaVal !== '') {
                $raw = get_post_meta($id, $metaKey, true);
                $rawStr = is_array($raw) ? wp_json_encode($raw) : (string)$raw;
                $pass = true;
                switch ($metaOp) {
                    case 'not_equals':
                        $pass = $rawStr !== $metaVal;
                        break;
                    case 'contains':
                        $pass = stripos($rawStr, $metaVal) !== false;
                        break;
                    case 'gt':
                        $pass = floatval($rawStr) > floatval($metaVal);
                        break;
                    case 'gte':
                        $pass = floatval($rawStr) >= floatval($metaVal);
                        break;
                    case 'lt':
                        $pass = floatval($rawStr) < floatval($metaVal);
                        break;
                    case 'lte':
                        $pass = floatval($rawStr) <= floatval($metaVal);
                        break;
                    case 'equals':
                    default:
                        $pass = $rawStr === $metaVal;
                        break;
                }
                if (!$pass) continue;
            }
            $out[] = $id;
        }
        return $out;
    }

    private function collect_record($id, $pt, $fields) {
        $post = get_post($id);
        if (!$post) return [];
        $rec = [];

        foreach ($fields as $field) {
            if ($field === 'ID')       { $rec['ID'] = $post->ID; continue; }
            if ($field === 'title')    { $rec['title'] = get_the_title($post); continue; }
            if ($field === 'slug')     { $rec['slug'] = $post->post_name; continue; }
            if ($field === 'content')  { $rec['content'] = $post->post_content; continue; }
            if ($field === 'excerpt')  { $rec['excerpt'] = $post->post_excerpt; continue; }
            if ($field === 'status')   { $rec['status'] = $post->post_status; continue; }
            if ($field === 'date')     { $rec['date'] = $post->post_date_gmt; continue; }
            if ($field === 'modified') { $rec['modified'] = $post->post_modified_gmt; continue; }
            if ($field === 'author')   { $rec['author'] = $post->post_author; continue; }
            if ($field === 'link')     { $rec['link'] = get_permalink($post); continue; }

            if (str_starts_with($field, 'tax:')) {
                $tax = substr($field, 4);
                $terms = wp_get_post_terms($post->ID, $tax, ['fields'=>'all']);
                if (!is_wp_error($terms)) {
                    $rec['tax'][$tax] = array_map(function($t){
                        return ['term_id'=>$t->term_id, 'slug'=>$t->slug, 'name'=>$t->name];
                    }, $terms);
                }
                continue;
            }

            if (str_starts_with($field, 'acf:')) {
                $name = substr($field, 4);
                if (function_exists('get_field')) {
                    $rec['acf'][$name] = get_field($name, $post->ID);
                } else {
                    $rec['acf'][$name] = get_post_meta($post->ID, $name, true);
                }
                continue;
            }

            // meta
            $rec[$field] = get_post_meta($post->ID, $field, true);
        }

        if ($pt === 'product' && in_array('_product_attributes', $fields, true)) {
            $rec['_product_attributes'] = get_post_meta($post->ID, '_product_attributes', true);
        }
        return $rec;
    }

    private function write_json($dir, $file_base, $pt, $payload) {
        otto_db_ensure_output_dir($dir);
        $path = trailingslashit($dir) . $file_base . '.json';
        file_put_contents($path, wp_json_encode($payload, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE));
        return $path;
    }
}
