<?php
if (!defined('ABSPATH')) exit;

class OTTO_Data_Binder_REST {
    private $option_key;
    private $cron_base;
    const REST_NS = 'otto-data-binder/v1';

    public function __construct($option_key, $cron_base) {
        $this->option_key = $option_key;
        $this->cron_base  = $cron_base;
    }

    public function hooks() {
        add_action('rest_api_init', [$this, 'register']);
    }

    public function register() {
        // Pretty file URL: /wp-json/otto-data-binder/v1/file/{slug}.json
        register_rest_route(self::REST_NS, '/file/(?P<slug>[^\/]+)\.json', [
            'methods'  => 'GET',
            'callback' => [$this, 'get_file'],
            'permission_callback' => '__return_true',
            'args' => [
                'slug' => ['required' => true, 'type' => 'string'],
                'key'  => ['required' => false, 'type' => 'string'],
            ],
        ]);

        // Generate-by-URL: /wp-json/otto-data-binder/v1/generate/{slug}
        register_rest_route(self::REST_NS, '/generate/(?P<slug>[^\/]+)', [
            'methods'  => 'GET',
            'callback' => [$this, 'generate_now'],
            'permission_callback' => '__return_true',
            'args' => [
                'slug' => ['required' => true, 'type' => 'string'],
                'key'  => ['required' => false, 'type' => 'string'],
            ],
        ]);
    }

    public function get_file(WP_REST_Request $req) {
        $settings = otto_db_get_settings($this->option_key);
        $slug = sanitize_title_with_dashes($req->get_param('slug'));
        $tpl = $this->find_template($settings['templates'], $slug);
        if (!$tpl) return new WP_REST_Response(['error'=>'unknown template'], 404);
        if (($tpl['enabled'] ?? 'yes') === 'no') {
            return new WP_REST_Response(['error' => 'template disabled'], 404);
        }

        if (($tpl['secret_enabled'] ?? 'yes') === 'yes') {
            $key = (string)$req->get_param('key');
            if (!$key || !hash_equals($tpl['secret_key'], $key)) {
                return new WP_REST_Response(['error'=>'unauthorized'], 401);
            }
        }

        $path = trailingslashit($settings['output_dir']) . $slug . '.json';
        if (!file_exists($path)) return new WP_REST_Response(['error'=>'file not found'], 404);

        $content = file_get_contents($path);
        return new WP_REST_Response($content, 200, ['Content-Type'=>'application/json']);
    }

    public function generate_now(WP_REST_Request $req) {
        $settings = otto_db_get_settings($this->option_key);
        $slug = sanitize_title_with_dashes($req->get_param('slug'));
        $tpl = $this->find_template($settings['templates'], $slug);
        if (!$tpl) return new WP_REST_Response(['error'=>'unknown template'], 404);
        if (($tpl['enabled'] ?? 'yes') === 'no') {
            return new WP_REST_Response(['error' => 'template disabled'], 404);
        }

        if (($tpl['secret_enabled'] ?? 'yes') === 'yes') {
            $key = (string)$req->get_param('key');
            if (!$key || !hash_equals($tpl['secret_key'], $key)) {
                return new WP_REST_Response(['error'=>'unauthorized'], 401);
            }
        }

        // Trigger generation immediately
        $hook = otto_db_event_name($this->cron_base, $slug);
        do_action($hook, $slug);

        return new WP_REST_Response(['ok'=>true,'slug'=>$slug], 200);
    }

    private function find_template($templates, $slug) {
        foreach ($templates as $t) if ($t['slug'] === $slug) return $t;
        return null;
    }
}
