<?php
if (!defined('ABSPATH')) exit;

class Suggent_Bot_Panel_Admin {
    private $option_key;

    public function __construct($option_key) {
        $this->option_key = $option_key;
    }

    public function hooks() {
        add_action('admin_menu', [$this, 'menu']);
        add_action('admin_init', [$this, 'handle_post']);
        add_action('admin_enqueue_scripts', [$this, 'assets']);
        add_action('wp_ajax_suggent_bp_test', [$this, 'ajax_test']);
        add_action('wp_ajax_suggent_bp_fetch_bots', [$this, 'ajax_fetch_bots']);
    }

    public function menu() {
        add_menu_page(
            'Suggent Product Suggestions',
            'Suggent',
            'manage_options',
            SUGGENT_BP_SLUG,
            [$this, 'render'],
            'dashicons-format-chat',
            82
        );

    }

    public function assets($hook) {
        if ($hook !== 'toplevel_page_' . SUGGENT_BP_SLUG) return;
        wp_enqueue_style('suggent-bp-admin', SUGGENT_BP_URL . 'assets/admin.css', [], SUGGENT_BP_VERSION);
        wp_enqueue_script('suggent-bp-admin', SUGGENT_BP_URL . 'assets/admin.js', ['jquery'], SUGGENT_BP_VERSION, true);
        wp_localize_script('suggent-bp-admin', 'SUGGENT_BP', [
            'nonce' => wp_create_nonce('suggent_bp_nonce'),
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'strings' => [
                'testError' => __('Connection failed.', 'suggent-bot-panel'),
                'botError' => __('Unable to load bots.', 'suggent-bot-panel'),
            ],
        ]);
    }

    public function handle_post() {
        if (!current_user_can('manage_options')) return;
        if (!isset($_POST['suggent_bp_save']) || !check_admin_referer('suggent_bp_save', 'suggent_bp_nonce')) return;

        $settings = suggent_bp_get_settings($this->option_key);
        $postedKey = sanitize_text_field((string)($_POST['widget_key'] ?? ''));
        if ($postedKey !== '') {
            $settings['widget_key'] = base64_encode($postedKey);
        }
        $settings['default_bot_id'] = sanitize_text_field((string)($_POST['default_bot_id'] ?? ''));
        $settings['theme_color'] = sanitize_hex_color((string)($_POST['theme_color'] ?? '#22c7a6')) ?: '#22c7a6';
        $settings['position'] = in_array(($_POST['position'] ?? 'right'), ['left','right'], true) ? $_POST['position'] : 'right';
        $settings['global_embed'] = isset($_POST['global_embed']) ? 'yes' : 'no';

        suggent_bp_save_settings($this->option_key, $settings);
        add_settings_error('suggent_bp', 'settings_saved', 'Settings saved.', 'updated');
    }

    public function ajax_test() {
        check_ajax_referer('suggent_bp_nonce', 'nonce');
        if (!current_user_can('manage_options')) wp_send_json_error(['message' => 'Forbidden']);

        $settings = suggent_bp_get_settings($this->option_key);
        $panel = suggent_bp_panel_url($settings);
        $decoded = $settings['widget_key'] ? base64_decode((string)$settings['widget_key']) : '';
        [$ok, $data] = suggent_bp_call_panel($panel, '/public/widget/bootstrap.php', $decoded);
        if (!$ok) {
            wp_send_json_error(['message' => is_string($data) ? $data : 'Connection failed']);
        }
        wp_send_json_success(['data' => $data]);
    }

    public function ajax_fetch_bots() {
        check_ajax_referer('suggent_bp_nonce', 'nonce');
        if (!current_user_can('manage_options')) wp_send_json_error(['message' => 'Forbidden']);

        $settings = suggent_bp_get_settings($this->option_key);
        $panel = suggent_bp_panel_url($settings);
        $decoded = $settings['widget_key'] ? base64_decode((string)$settings['widget_key']) : '';
        [$ok, $data] = suggent_bp_call_panel($panel, '/public/widget/bots.php', $decoded);
        if (!$ok) {
            wp_send_json_error(['message' => is_string($data) ? $data : 'Unable to load bots']);
        }
        wp_send_json_success(['data' => $data]);
    }

    public function render() {
        $settings = suggent_bp_get_settings($this->option_key);
        $panelUrl = suggent_bp_panel_url($settings);
        $connected = !empty($settings['widget_key']);
        $decodedKey = $connected ? base64_decode((string)$settings['widget_key']) : '';
        $tab = isset($_GET['tab']) ? sanitize_text_field((string)$_GET['tab']) : 'dashboard';
        $validTabs = ['dashboard', 'settings', 'bots'];
        if (!in_array($tab, $validTabs, true)) {
            $tab = 'dashboard';
        }

        $botsData = null;
        $accountData = null;
        if ($connected && ($tab === 'dashboard' || $tab === 'bots')) {
            [$ok, $data] = suggent_bp_call_panel($panelUrl, '/public/widget/bots.php', $decodedKey);
            if ($ok && is_array($data)) {
                $botsData = $data;
            }
        }
        if ($connected && $tab === 'dashboard') {
            [$ok, $data] = suggent_bp_call_panel($panelUrl, '/public/widget/account.php', $decodedKey);
            if ($ok && is_array($data)) {
                $accountData = $data;
            }
        }

        settings_errors('suggent_bp');
        ?>
        <div class="wrap">
            <div class="suggent-bp-about">
                <div class="suggent-bp-about-left">
                    <div class="suggent-bp-logo">S</div>
                </div>
                <div class="suggent-bp-about-right">
                    <h1 class="suggent-bp-title">Suggent Product Suggestions</h1>
                    <p class="suggent-bp-sub">Connect your WordPress site to Suggent bots.</p>
                </div>
            </div>

            <nav class="suggent-bp-tabs">
                <a href="<?php echo esc_url(admin_url('admin.php?page=' . SUGGENT_BP_SLUG . '&tab=dashboard')); ?>" class="<?php echo $tab === 'dashboard' ? 'active' : ''; ?>">Dashboard</a>
                <a href="<?php echo esc_url(admin_url('admin.php?page=' . SUGGENT_BP_SLUG . '&tab=settings')); ?>" class="<?php echo $tab === 'settings' ? 'active' : ''; ?>">Settings</a>
                <a href="<?php echo esc_url(admin_url('admin.php?page=' . SUGGENT_BP_SLUG . '&tab=bots')); ?>" class="<?php echo $tab === 'bots' ? 'active' : ''; ?>">Bots</a>
                <a href="<?php echo esc_url(admin_url('admin.php?page=' . SUGGENT_DB_SLUG)); ?>">Data Tool</a>
            </nav>

            <?php if ($tab === 'dashboard'): ?>
                <div class="suggent-bp-card">
                    <h2>Overview</h2>
                    <?php if (!$connected): ?>
                        <p><strong>Connect your Suggent widget to get started.</strong></p>
                        <p>Go to the Settings tab to add your Widget Key.</p>
                    <?php else: ?>
                        <?php
                          $bots = is_array($botsData['bots'] ?? null) ? $botsData['bots'] : [];
                          $totalBots = count($bots);
                          $activeBots = 0;
                          $defaultBot = '';
                          foreach ($bots as $b) {
                              if (!empty($b['active'])) $activeBots++;
                              if (!empty($b['is_default'])) $defaultBot = (string)($b['name'] ?? '');
                          }
                        ?>
                        <div class="suggent-bp-stats">
                            <div class="suggent-bp-stat">
                                <div class="label">Total Bots</div>
                                <div class="value"><?php echo esc_html($totalBots); ?></div>
                            </div>
                            <div class="suggent-bp-stat">
                                <div class="label">Active Bots</div>
                                <div class="value"><?php echo esc_html($activeBots); ?></div>
                            </div>
                            <div class="suggent-bp-stat">
                                <div class="label">Default Bot</div>
                                <div class="value"><?php echo esc_html($defaultBot !== '' ? $defaultBot : 'Not set'); ?></div>
                            </div>
                            <div class="suggent-bp-stat">
                                <div class="label">Credit Balance</div>
                                <div class="value"><?php echo esc_html((string)($accountData['balance'] ?? '—')); ?></div>
                            </div>
                        </div>
                        <div style="margin-top:12px;">
                            <a class="button suggent-bp-btn suggent-bp-btn-panel" href="<?php echo esc_url(rtrim($panelUrl, '/') . '/public/suggent/bots.php'); ?>" target="_blank" rel="noopener">Open Suggent Panel</a>
                            <a class="button suggent-bp-btn suggent-bp-btn-credit" href="<?php echo esc_url(rtrim($panelUrl, '/') . '/public/deposit.php'); ?>" target="_blank" rel="noopener">Buy Credits</a>
                        </div>
                    <?php endif; ?>
                </div>
            <?php endif; ?>

            <?php if ($tab === 'settings'): ?>
            <form method="post" class="suggent-bp-form">
                <?php wp_nonce_field('suggent_bp_save', 'suggent_bp_nonce'); ?>
                <input type="hidden" name="suggent_bp_save" value="1" />

                <h2 class="title">Step 1 — Connect Your Suggent Account</h2>
                <table class="form-table">
                    <tr>
                        <th scope="row"><label for="widget_key">Widget Key</label></th>
                        <td>
                            <input name="widget_key" id="widget_key" type="password" class="regular-text" value="" autocomplete="new-password" placeholder="<?php echo $connected ? 'Saved' : 'Paste your key'; ?>" />
                            <p class="description">Paste your widget key to connect. It is stored encoded.</p>
                        </td>
                    </tr>
                </table>

                <?php if ($connected): ?>
                    <h2 class="title">Step 2 — Choose Your Default Bot</h2>
                    <table class="form-table">
                        <tr>
                            <th scope="row">Default Bot</th>
                            <td>
                                <select name="default_bot_id" id="suggent_bp_default_bot" data-current="<?php echo esc_attr($settings['default_bot_id']); ?>">
                                    <option value="">-- Select a bot --</option>
                                </select>
                                <p class="description">Use “Refresh Bots” to load from your Suggent account.</p>
                                <button type="button" class="button" id="suggent_bp_refresh_bots">Refresh Bots</button>
                            </td>
                        </tr>
                    </table>

                    <h2 class="title">Step 3 — Widget Settings</h2>
                    <table class="form-table">
                        <tr>
                            <th scope="row"><label for="theme_color">Theme Color</label></th>
                            <td><input name="theme_color" id="theme_color" type="text" class="regular-text" value="<?php echo esc_attr($settings['theme_color']); ?>" /></td>
                        </tr>
                        <tr>
                            <th scope="row">Position</th>
                            <td>
                                <label><input type="radio" name="position" value="right" <?php checked($settings['position'], 'right'); ?> /> Right</label>
                                <label style="margin-left:12px;"><input type="radio" name="position" value="left" <?php checked($settings['position'], 'left'); ?> /> Left</label>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row">Global Embed</th>
                            <td><label><input type="checkbox" name="global_embed" <?php checked($settings['global_embed'], 'yes'); ?> /> Embed widget site-wide</label></td>
                        </tr>
                    </table>
                <?php else: ?>
                    <p><strong>Connect your widget to continue.</strong> Enter a Widget Key and save to unlock bot selection and widget settings.</p>
                <?php endif; ?>

                <p>
                    <button type="submit" class="button button-primary">Save Settings</button>
                    <button type="button" class="button" id="suggent_bp_test">Test Connection</button>
                    <span id="suggent_bp_test_result" class="suggent-bp-test"></span>
                </p>

                <?php if ($connected): ?>
                    <h2 class="title">Step 4 — Embed</h2>
                    <p>Use <code>[suggent_chat]</code> to embed the widget on a specific page.</p>
                <?php endif; ?>
            </form>
            <?php endif; ?>

            <?php if ($tab === 'bots'): ?>
                <div class="suggent-bp-card">
                    <h2>Bots</h2>
                    <?php if (!$connected): ?>
                        <p><strong>Connect your widget first.</strong> Go to the Settings tab and add your Widget Key.</p>
                    <?php else: ?>
                        <?php $bots = is_array($botsData['bots'] ?? null) ? $botsData['bots'] : []; ?>
                        <?php if (!$bots): ?>
                            <p>No bots found. Create one in the Suggent panel.</p>
                        <?php else: ?>
                            <table class="widefat striped">
                                <thead>
                                    <tr>
                                        <th>Bot</th>
                                        <th>Status</th>
                                        <th>Default</th>
                                        <th></th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($bots as $bot): ?>
                                        <tr>
                                            <td><?php echo esc_html((string)($bot['name'] ?? 'Bot')); ?></td>
                                            <td><?php echo !empty($bot['active']) ? 'Active' : 'Inactive'; ?></td>
                                            <td><?php echo !empty($bot['is_default']) ? 'Yes' : 'No'; ?></td>
                                            <td style="text-align:right;">
                                                <a class="button" href="<?php echo esc_url(rtrim($panelUrl, '/') . '/public/suggent/bots.php'); ?>" target="_blank" rel="noopener">Manage</a>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        <?php endif; ?>
                        <p style="margin-top:12px;">
                            <a class="button suggent-bp-btn suggent-bp-btn-panel" href="<?php echo esc_url(rtrim($panelUrl, '/') . '/public/suggent/bots.php'); ?>" target="_blank" rel="noopener">Open Suggent Panel</a>
                        </p>
                    <?php endif; ?>
                </div>
            <?php endif; ?>
        </div>
        <?php
    }
}
