<?php
if (!defined('ABSPATH')) exit;

define('OTTO_DB_SCHEMA_VERSION', '2'); // strict schema for templates

function otto_db_defaults() {
    return [
        'schema_version' => OTTO_DB_SCHEMA_VERSION,
        'output_dir'     => '',         // uploads/otto-binders
        'templates'      => [],         // array of template arrays (see otto_db_template_defaults)
        // uninstall behavior
        'cleanup_on_uninstall'        => 'yes',
        'delete_binders_on_uninstall' => 'no',
    ];
}

function otto_db_template_defaults() {
    return [
        'enabled'        => 'yes',         // yes|no
        'slug'            => '',             // file base name (e.g., "product-price")
        'label'           => '',             // admin label
        'post_type'       => 'post',
        'fields'          => [],             // array of field keys ('ID','title', meta keys, 'acf:field', 'tax:taxonomy')
        'filters'         => [
            'post_status'   => 'publish',    // publish|any
            'only_in_stock' => 'no',         // yes|no (for Woo products)
            'sku_whitelist' => '',
            'sku_blacklist' => '',
            'category_slugs' => '',          // CSV of product_cat slugs
            'tag_slugs'      => '',          // CSV of product_tag slugs
            'meta_key'       => '',          // meta key to filter
            'meta_op'        => 'equals',    // equals|not_equals|contains|gt|gte|lt|lte
            'meta_value'     => '',          // meta value
        ],
        'schedule'        => 'manual',       // manual|hourly|twicedaily|daily
        'secret_enabled'  => 'yes',
        'secret_key'      => '',
        'last_run'        => [ 'time'=>0, 'bytes'=>0, 'path'=>'' ],
    ];
}

/** STRICT loader: resets if schema mismatch; no migration of renamed keys. */
function otto_db_get_settings($option_key) {
    $stored = get_option($option_key, null);
    if (!is_array($stored) || !isset($stored['schema_version']) || $stored['schema_version'] !== OTTO_DB_SCHEMA_VERSION) {
        $settings = otto_db_defaults();
        $uploads = wp_get_upload_dir();
        $settings['output_dir'] = trailingslashit($uploads['basedir']) . 'otto-binders';
        update_option($option_key, $settings, false);
        return $settings;
    }
    $defaults = otto_db_defaults();
    $settings = $defaults;
    foreach ($defaults as $k => $v) {
        if (array_key_exists($k, $stored)) $settings[$k] = $stored[$k];
    }
    if ($settings['output_dir'] === '') {
        $uploads = wp_get_upload_dir();
        $settings['output_dir'] = trailingslashit($uploads['basedir']) . 'otto-binders';
    }
    // sanitize templates
    $clean = [];
    if (is_array($settings['templates'])) {
        foreach ($settings['templates'] as $tpl) {
            $tdef = otto_db_template_defaults();
            $t = $tdef;
            foreach ($tdef as $k=>$v) {
                if (isset($tpl[$k])) {
                    $t[$k] = $tpl[$k];
                }
            }
            if ($t['secret_key'] === '') $t['secret_key'] = wp_generate_password(24, false, false);
            $t['enabled'] = ($t['enabled'] === 'no') ? 'no' : 'yes';
            $clean[] = $t;
        }
    }
    $settings['templates'] = $clean;
    return $settings;
}

function otto_db_save_settings($option_key, $settings) {
    // enforce schema; drop unknown keys
    $clean = [];
    foreach (otto_db_defaults() as $k => $v) {
        $clean[$k] = array_key_exists($k, $settings) ? $settings[$k] : $v;
    }
    // clean templates
    $out = [];
    foreach ($clean['templates'] as $tpl) {
        $tdef = otto_db_template_defaults();
        $t = [];
        foreach ($tdef as $k=>$v) {
            $t[$k] = array_key_exists($k,$tpl) ? $tpl[$k] : $v;
        }
        if ($t['secret_key'] === '') $t['secret_key'] = wp_generate_password(24, false, false);
        $t['enabled'] = ($t['enabled'] === 'no') ? 'no' : 'yes';
        $out[] = $t;
    }
    $clean['templates'] = $out;
    $clean['schema_version'] = OTTO_DB_SCHEMA_VERSION;
    update_option($option_key, $clean, false);
}

function otto_db_ensure_output_dir($dir) {
    if (!file_exists($dir)) wp_mkdir_p($dir);
    if (is_writable($dir)) {
        $ht = $dir . '/.htaccess';
        if (!file_exists($ht)) {
            @file_put_contents($ht, "Options -Indexes\n<FilesMatch \"\\.(json)$\">\n  Require all denied\n</FilesMatch>\n");
        }
        $wc = $dir . '/web.config';
        if (!file_exists($wc)) {
            @file_put_contents($wc, "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n<configuration>\n<system.webServer>\n  <directoryBrowse enabled=\"false\" />\n  <security>\n    <requestFiltering>\n      <fileExtensions>\n        <add fileExtension=\".json\" allowed=\"false\" />\n      </fileExtensions>\n    </requestFiltering>\n  </security>\n</system.webServer>\n</configuration>");
        }
        $idx = $dir . '/index.html';
        if (!file_exists($idx)) @file_put_contents($idx, "");
    }
}

function otto_db_human_bytes($bytes) {
    $bytes = (int)$bytes;
    if ($bytes < 1024) return $bytes . ' B';
    $units = ['KB','MB','GB','TB'];
    $i = 0; $val = $bytes / 1024;
    while ($val >= 1024 && $i < count($units)-1) { $val /= 1024; $i++; }
    return number_format_i18n($val, 2) . ' ' . $units[$i];
}

function otto_db_csv_to_array($csv) {
    $csv = trim((string)$csv);
    if ($csv === '') return [];
    $parts = array_map('trim', explode(',', $csv));
    return array_values(array_filter($parts, fn($v) => $v !== ''));
}

function otto_db_get_public_post_types() {
    return get_post_types(['public' => true], 'objects');
}

function otto_db_core_fields_for($pt) {
    $common = ['ID','title','slug','content','excerpt','status','date','modified','author','link'];
    if ($pt === 'product') {
        $woo = ['_sku','_price','_regular_price','_sale_price','_stock','_stock_status','_manage_stock','_backorders','_weight','_length','_width','_height','_tax_class','_product_attributes'];
        return array_merge($common, $woo);
    }
    return $common;
}

function otto_db_taxonomies_for($pt) {
    $taxes = get_object_taxonomies($pt, 'objects');
    $out = [];
    foreach ($taxes as $slug => $tax) $out[$slug] = $tax->label . " ($slug)";
    ksort($out, SORT_NATURAL | SORT_FLAG_CASE);
    return $out;
}

function otto_db_detect_meta_keys_for($pt, $limit=300) {
    global $wpdb;
    $keys = [];
    $sql = $wpdb->prepare("
        SELECT DISTINCT pm.meta_key
        FROM {$wpdb->postmeta} pm
        INNER JOIN {$wpdb->posts} p ON p.ID = pm.post_id
        WHERE p.post_type = %s
        LIMIT %d
    ", $pt, $limit);
    $rows = $wpdb->get_col($sql);
    if (is_array($rows)) $keys = array_merge($keys, $rows);
    $keys = array_values(array_unique(array_filter($keys)));
    sort($keys, SORT_NATURAL | SORT_FLAG_CASE);
    return $keys;
}

function otto_db_acf_fields_for($pt) {
    $out = [];
    if (!function_exists('acf_get_field_groups') || !function_exists('acf_get_fields')) return $out;
    $groups = acf_get_field_groups(['post_type' => $pt]);
    foreach ($groups as $group) {
        $fields = acf_get_fields($group['key']);
        if (!$fields) continue;
        foreach ($fields as $f) {
            if (!empty($f['name'])) {
                $name = (string)$f['name'];
                $label = !empty($f['label']) ? $f['label'] : $name;
                $out['acf:' . $name] = $label . ' (' . $name . ')';
            }
        }
    }
    ksort($out, SORT_NATURAL | SORT_FLAG_CASE);
    return $out;
}

/** Cron helpers */
function otto_db_event_name($base, $slug) { return $base . sanitize_title_with_dashes($slug); }

function otto_db_reschedule_all_templates($base, $templates) {
    otto_db_clear_all_template_crons($base);
    foreach ($templates as $t) {
        if (empty($t['slug']) || ($t['enabled'] ?? 'yes') === 'no') continue;
        $ev = otto_db_event_name($base, $t['slug']);
        if ($t['schedule'] !== 'manual' && !wp_next_scheduled($ev)) {
            wp_schedule_event(time() + 60, $t['schedule'], $ev, [$t['slug']]);
        }
    }
}

function otto_db_clear_all_template_crons($base) {
    // There’s no registry; clear common recurrences by scanning scheduled events for hooks we used.
    $crons = _get_cron_array();
    if (!is_array($crons)) return;
    foreach ($crons as $ts => $events) {
        foreach ($events as $hook => $d) {
            if (str_starts_with($hook, $base)) {
                wp_clear_scheduled_hook($hook);
            }
        }
    }
}
